const express = require('express');
const bodyParser = require('body-parser');
const dotenv = require('dotenv');
const authRoutes = require('./handlers/auth');
const streamRoutes = require('./handlers/streams');
const giftRoutes = require('./handlers/gifts');
const coinRoutes = require('./handlers/coins');
const pool = require('./db');

dotenv.config();

const app = express();
app.use(bodyParser.json());

async function initDb() {
    const queries = [
        `CREATE TABLE IF NOT EXISTS users (
            id BIGINT PRIMARY KEY AUTO_INCREMENT,
            username VARCHAR(255) UNIQUE NOT NULL,
            password_hash VARCHAR(255) NOT NULL,
            coins BIGINT DEFAULT 0 NOT NULL
        );`,
        `CREATE TABLE IF NOT EXISTS live_streams (
            id BIGINT PRIMARY KEY AUTO_INCREMENT,
            stream_id VARCHAR(36) UNIQUE NOT NULL,
            host_id BIGINT NOT NULL,
            title VARCHAR(255) NOT NULL,
            is_live BOOLEAN DEFAULT FALSE,
            FOREIGN KEY (host_id) REFERENCES users(id)
        );`,
        `CREATE TABLE IF NOT EXISTS gift_transactions (
            id BIGINT PRIMARY KEY AUTO_INCREMENT,
            sender_id BIGINT NOT NULL,
            receiver_stream_id VARCHAR(36) NOT NULL,
            gift_type BIGINT NOT NULL,
            gift_name VARCHAR(255) NOT NULL,
            coin_cost BIGINT NOT NULL,
            amount BIGINT NOT NULL,
            timestamp BIGINT NOT NULL,
            FOREIGN KEY (sender_id) REFERENCES users(id),
            FOREIGN KEY (receiver_stream_id) REFERENCES live_streams(stream_id)
        );`
    ];

    const connection = await pool.getConnection();
    try {
        for (const query of queries) {
            await connection.query(query);
        }
    } finally {
        connection.release();
    }
}

app.use('/auth', authRoutes);
app.use('/streams', streamRoutes);
app.use('/gifts', giftRoutes);
app.use('/coins', coinRoutes);

app.listen(8080, async () => {
    await initDb();
    console.log('Server running at http://0.0.0.0:8080');
});
