const express = require('express');
const { v4: uuidv4 } = require('uuid');
const router = express.Router();
const pool = require('../db');

router.post('/start', async (req, res) => {
    const { title, user_id } = req.body;
    if (!title || title.trim() === '') {
        return res.status(400).send('Stream title is required');
    }

    const stream_id = uuidv4();
    try {
        await pool.query('INSERT INTO live_streams (stream_id, host_id, title, is_live) VALUES (?, ?, ?, false)', [stream_id, user_id, title]);
        res.status(200).json({ stream_id, user_id, message: 'Live stream is in preview mode' });
    } catch (error) {
        res.status(500).send('Database error');
    }
});

router.post('/confirm', async (req, res) => {
    const { stream_id } = req.body;
    if (!stream_id) {
        return res.status(400).send('stream_id is required');
    }

    try {
        const [result] = await pool.query('UPDATE live_streams SET is_live = true WHERE stream_id = ?', [stream_id]);
        if (result.affectedRows > 0) {
            res.status(200).json({ stream_id, is_live: true, message: 'Live stream confirmed and is now live' });
        } else {
            res.status(404).send('Stream not found');
        }
    } catch (error) {
        res.status(500).send('Database error');
    }
});

router.get('/live', async (req, res) => {
    try {
        const [streams] = await pool.query(`
            SELECT ls.stream_id, ls.host_id, u.username AS host_username, ls.title, ls.is_live
            FROM live_streams ls
            JOIN users u ON ls.host_id = u.id
            WHERE ls.is_live = true
            ORDER BY ls.id DESC
        `);
        res.status(200).json({ streams });
    } catch (error) {
        res.status(500).send('Database error');
    }
});

router.post('/join', async (req, res) => {
    const { stream_id, role, user_id } = req.body;
    if (!stream_id || !role || !user_id) {
        return res.status(400).send('stream_id, role, and user_id are required');
    }

    const validRoles = ['audience', 'co-host'];
    if (!validRoles.includes(role.toLowerCase())) {
        return res.status(400).send("Role must be 'audience' or 'co-host'");
    }

    try {
        const [stream] = await pool.query('SELECT is_live FROM live_streams WHERE stream_id = ?', [stream_id]);
        if (stream.length === 0) {
            return res.status(404).send('Stream not found');
        }
        if (!stream[0].is_live) {
            return res.status(400).send('Stream is not live');
        }
        res.status(200).json({ message: `Joined stream as ${role}`, user: { user_id } });
    } catch (error) {
        res.status(500).send('Database error');
    }
});

router.post('/stop', async (req, res) => {
    const { host_id } = req.body;
    if (!host_id) {
        return res.status(400).send('host_id is required');
    }

    try {
        await pool.query('UPDATE live_streams SET is_live = false WHERE host_id = ?',[host_id]);
        res.status(200).send('Live stream stopped');
    } catch (error) {
        res.status(500).send('Database error');
    }
});

module.exports = router;
